% Generate the figures and tables from Katzin et al (2020) Biosystems Engineering:
%   Katzin, D., van Mourik, S., Kempkes, F., & van Henten, E. J. (2020). 
%   GreenLight  An open source model for greenhouses with supplemental lighting: 
%   Evaluation of heat requirements under LED and HPS lamps. 
%   Biosystems Engineering, 194, 6181. https://doi.org/10.1016/j.biosystemseng.2020.03.010
%
% This script file should be run from its directory, i.e., the active directory in
% MATLAB should be ...\Data from GreenLight (2020)\Simulation data\Source code

% David Katzin, Wageningen University & Research
% david.katzin@wur.nl
%
% Written in July 2023, reconstructing what was done for the paper back
% in 2019.

%% Set up directories
currentFilename = mfilename();
fullFilePath = mfilename('fullpath');
currentFileDir = strrep(fullFilePath, currentFilename, '');
if isempty(currentFileDir) || contains(currentFileDir, 'Temp\Editor') % This script is run from the console
    currentFileDir = cd(); % the directory from which this script is running
end

addpath(genpath([currentFileDir '\BramVanthoorModelVer2']));
addpath(genpath([currentFileDir '\StateSpaceModel']));

% Ensure currentFileDir ends with a  '\'
if currentFileDir(end) ~= '\'
    currentFileDir = [currentFileDir '\'];
end

currentDirSlashes = strfind(currentFileDir,'\');
simDir = currentFileDir(1:currentDirSlashes(end-1)); % Up one level from currentFileDir
                                                    % This should be the
                                                    % "Simualtion data\" directory
outputPath = [simDir 'MATLAB output\Replicated 2023'];

%% Load data
load([outputPath '\climateModel_hps_manuscriptParams.mat']);
hpsClimate = bv;

load([outputPath '\climateModel_led_manuscriptParams.mat']);
ledClimate = bv;

load([outputPath '\energyUse_hps__manuscriptParams_1-1.mat']);
hpsEnergy = bv;

load([outputPath '\energyUse_led__manuscriptParams_1-1.mat']);
ledEnergy = bv;

%% Table 1 - List of greenhouse parameters
paramList = [ ...
    string(hpsClimate.p.psi.label) num2str(hpsClimate.p.psi.val) 
    string(hpsClimate.p.aCov.label) num2str(hpsClimate.p.aCov.val) 
    string(hpsClimate.p.aFlr.label) num2str(hpsClimate.p.aFlr.val) 
    string(hpsClimate.p.hAir.label) num2str(hpsClimate.p.hAir.val) 
    string(hpsClimate.p.hGh.label) num2str(hpsClimate.p.hGh.val) 
    string(hpsClimate.p.aRoof.label) num2str(hpsClimate.p.aRoof.val) 
    string(hpsClimate.p.hVent.label) num2str(hpsClimate.p.hVent.val) 
    string(hpsClimate.p.cLeakage.label) num2str(hpsClimate.p.cLeakage.val) 
    string(hpsClimate.p.cDgh.label) num2str(hpsClimate.p.cDgh.val) 
    string(hpsClimate.p.cWgh.label) num2str(hpsClimate.p.cWgh.val) 
    string(hpsClimate.p.leakTop.label) num2str(hpsClimate.p.leakTop.val) 
    string(hpsClimate.p.tauThScrPar.label) num2str(hpsClimate.p.tauThScrPar.val) 
    string(hpsClimate.p.tauBlScrPar.label) num2str(hpsClimate.p.tauBlScrPar.val) 
    string(hpsClimate.p.kThScr.label) num2str(hpsClimate.p.kThScr.val) 
    string(hpsClimate.p.kBlScr.label) num2str(hpsClimate.p.kBlScr.val) 
    string(hpsClimate.p.tauRfPar.label) num2str(hpsClimate.p.tauRfPar.val) 
    string(hpsClimate.p.tauRfNir.label) num2str(hpsClimate.p.tauRfNir.val) 
    string(hpsClimate.p.cHecIn.label) num2str(hpsClimate.p.cHecIn.val) 
    string(hpsClimate.p.phiExtCo2.label) num2str(hpsClimate.p.phiExtCo2.val) 
    string(hpsClimate.p.phiPipeE.label) num2str(hpsClimate.p.phiPipeE.val) 
    string(hpsClimate.p.phiPipeI.label) num2str(hpsClimate.p.phiPipeI.val) 
    string(hpsClimate.p.lPipe.label) num2str(hpsClimate.p.lPipe.val) 
    string(hpsClimate.p.phiGroPipeE.label) num2str(hpsClimate.p.phiGroPipeE.val) 
    string(hpsClimate.p.phiGroPipeI.label) num2str(hpsClimate.p.phiGroPipeI.val) 
    string(hpsClimate.p.lGroPipe.label) num2str(hpsClimate.p.lGroPipe.val) 
    ];

disp(paramList);
    
% Check that parameters were the same in both HPS model runs
compareParams(hpsClimate, hpsEnergy)
% Note that in the energy use model, pBoil and pGroBoil were set in such a
% way that there is no limit on how much energy is provided by the pipes


%% Table 2 - List of lamp parameters
lampParamList = [" "                        "HPS"                                   "LED"
    string(hpsClimate.p.lampsMax.label)     num2str(hpsClimate.p.lampsMax.val)      num2str(ledClimate.p.lampsMax.val) 
    string(hpsClimate.p.aLamp.label)        num2str(hpsClimate.p.aLamp.val)         num2str(ledClimate.p.aLamp.val) 
    string(hpsClimate.p.tauLampPar.label)   num2str(hpsClimate.p.tauLampPar.val)    num2str(ledClimate.p.tauLampPar.val) 
    string(hpsClimate.p.rhoLampPar.label)   num2str(hpsClimate.p.rhoLampPar.val)    num2str(ledClimate.p.rhoLampPar.val) 
    string(hpsClimate.p.tauLampPar.label)   num2str(hpsClimate.p.tauLampPar.val)    num2str(ledClimate.p.tauLampPar.val) 
    string(hpsClimate.p.rhoLampNir.label)   num2str(hpsClimate.p.rhoLampNir.val)    num2str(ledClimate.p.rhoLampNir.val) 
    string(hpsClimate.p.etaLampPar.label)   num2str(hpsClimate.p.etaLampPar.val)    num2str(ledClimate.p.etaLampPar.val) 
    string(hpsClimate.p.etaLampNir.label)   num2str(hpsClimate.p.etaLampNir.val)    num2str(ledClimate.p.etaLampNir.val) 
    string(hpsClimate.p.epsLampTop.label)   num2str(hpsClimate.p.epsLampTop.val)    num2str(ledClimate.p.epsLampTop.val) 
    string(hpsClimate.p.epsLampBottom.label) num2str(hpsClimate.p.epsLampBottom.val) num2str(ledClimate.p.epsLampBottom.val) 
    string(hpsClimate.p.capLamp.label)     num2str(hpsClimate.p.capLamp.val)      num2str(ledClimate.p.capLamp.val) 
    string(hpsClimate.p.lampCool.label)     num2str(hpsClimate.p.lampCool.val)      num2str(ledClimate.p.lampCool.val) 
    string(hpsClimate.p.hecLampAir.label)   num2str(hpsClimate.p.hecLampAir.val)    num2str(ledClimate.p.hecLampAir.val) 
    string(hpsClimate.p.parJtoUmolLamp.label) num2str(hpsClimate.p.parJtoUmolLamp.val) num2str(ledClimate.p.parJtoUmolLamp.val)   
    "Efficacy"                               num2str(hpsClimate.p.parJtoUmolLamp.val*hpsClimate.p.etaLampPar.val) ...
                                                num2str(ledClimate.p.parJtoUmolLamp.val*ledClimate.p.etaLampPar.val) 
    "PPFD"                                  num2str(hpsClimate.p.parJtoUmolLamp.val*hpsClimate.p.etaLampPar.val*hpsClimate.p.lampsMax.val) ...
                                                num2str(ledClimate.p.parJtoUmolLamp.val*ledClimate.p.etaLampPar.val*ledClimate.p.lampsMax.val)
    ];

disp(lampParamList);
compareParams(hpsClimate, hpsEnergy)
compareParams(ledClimate, ledEnergy)

%% Table 3 - model evaluation

% HPS climate model
load([outputPath '\climateModel_hps_manuscriptParams.mat']);
rmseRrmseMe;
hpsClimateResults = [""                     "HPS"
            "RMSE Tair (C)"        num2str(rmseTair,3)
            "RMSE RHair (%)"        num2str(rmseRhair,3)
            "RMSE CO2airppm (ppm)"  num2str(rmsePpm,3)
            "RRMSE Tair (%)"        num2str(100*rrmseTair,3)
            "RRMSE RHair (%)"       num2str(100*rrmseRhair,3)
            "RRMSE CO2airppm (%)"   num2str(100*rrmsePpm,3)
            "ME Tair (C)"          num2str(meTair,1)
            "ME RHair (%)"          num2str(meRhair,3)
            "ME CO2airppm (ppm)"    num2str(mePpm,3)     ];
disp(hpsClimateResults);

% LED climate model
load([outputPath '\climateModel_led_manuscriptParams.mat']);
rmseRrmseMe;
ledClimateResults = [""                     "LED"
            "RMSE Tair (C)"        num2str(rmseTair,3)
            "RMSE RHair (%)"        num2str(rmseRhair,3)
            "RMSE CO2airppm (ppm)"  num2str(rmsePpm,3)
            "RRMSE Tair (%)"        num2str(100*rrmseTair,3)
            "RRMSE RHair (%)"       num2str(100*rrmseRhair,3)
            "RRMSE CO2airppm (%)"   num2str(100*rrmsePpm,3)
            "ME Tair (C)"          num2str(meTair,1)
            "ME RHair (%)"          num2str(meRhair,3)
            "ME CO2airppm (ppm)"    num2str(mePpm,3)     ];
disp(ledClimateResults);

% HPS energy model
eMeasHps = 300e-6*sum(hpsEnergy.d.uPipe.val(:,2)+hpsEnergy.d.uGroPipe.val(:,2));
eSimHps = 300e-6*sum(hpsEnergy.a.hBoilPipe.val(:,2)+hpsEnergy.a.hBoilGroPipe.val(:,2));
errorRatioHps = eSimHps./eMeasHps*100-100;

% LED energy model
eMeasLed = 300e-6*sum(ledEnergy.d.uPipe.val(:,2)+ledEnergy.d.uGroPipe.val(:,2));
eSimLed = 300e-6*sum(ledEnergy.a.hBoilPipe.val(:,2)+ledEnergy.a.hBoilGroPipe.val(:,2));
errorRatioLed = eSimLed./eMeasLed*100-100;


resultsEnergy = ...
    [""                         "HPS"                   "LED" 
    "Measured heating (MJ/m2)"  num2str(eMeasHps,3)     num2str(eMeasLed,3)
    "Simulated heating (MJ/m2)" num2str(eSimHps,3)      num2str(eSimLed,3)
    "RE heating (%)"            num2str(errorRatioHps,3) num2str(errorRatioLed,2)];

disp(resultsEnergy);
        
%% Figure 5 - measured and simulated heating
plotHeating;

%% Figure 6 - climate trajectories
plotTrajectories;

%% Table 4 - energy output from lamps
bv = hpsClimate;
lampOutput;

lampOutputHps = ...
    [""                         "HPS"                  
    "Total lamp input (MJ/m2)"  num2str(total,3)    
    "PAR output (%)"            num2str(100*par.val,3)     
    "NIR output (%)"            num2str(100*nir.val,3)     
    "FIR output (%)"            num2str(100*fir,3)     
    "Convective output (%)"     num2str(100*conv,3)     
    "Cooling (%)"               num2str(100*cool,3)     
    ];

disp(lampOutputHps);

bv = ledClimate;
lampOutput;

lampOutputHps = ...
    [""                         "LED"                  
    "Total lamp input (MJ/m2)"  num2str(total,3)    
    "PAR output (%)"            num2str(100*par.val,3)     
    "NIR output (%)"            num2str(100*nir.val,3)     
    "FIR output (%)"            num2str(100*fir,3)     
    "Convective output (%)"     num2str(100*conv,3)     
    "Cooling (%)"               num2str(100*cool,3)     
    ];

disp(lampOutputHps);

%% Figure 7 - lamp temperature
oneDayLampTemps;
